/************************************************************************
 * NAME:	label.c
 *
 * DESCR:	Routines that deal with the HDOS label.
 *
 * NOTES:	
 ************************************************************************/

#include "hdos.h"

/************************************************************************
 * NAME:	hdos_label_unpack()
 *
 * DESCR:	Given a buffer containing the label, load the given label
 *		structure with its contents.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- buffer is assumed to be filled and long enough
 ************************************************************************/
static int
hdos_label_unpack(struct hdos_label *label, unsigned char *buffer)
{
    label->serial = buffer[0];				/* disk serial number		*/

    hdos_date_unpack(					/* date disk initialized	*/
	hdos_int_unpack(&buffer[1]),&(label->initdate));

    label->dirsec = hdos_int_unpack(&buffer[3]);	/* logical sec where dir starts	*/
    label->grtsec = hdos_int_unpack(&buffer[5]);	/* logical sec where GRT starts	*/
    label->grpsize = buffer[7];				/* group size (2,4, or 8 secs)	*/
    label->voltype = buffer[8];				/* type of vol (0 = data,
							                1 = bootable,
									2 = no dir)	*/
    label->version = buffer[9];				/* version of init.abs for init	*/
    label->rgtsec = hdos_int_unpack(&buffer[10]);	/* logical sec where RGT starts	*/
    label->size = hdos_int_unpack(&buffer[12]);		/* number of sectors on disk	*/
    label->physize = hdos_int_unpack(&buffer[14]);	/* phys sector size (fixed 256)	*/
    label->flags = buffer[16];				/* volume flags 00 = 40 tk 1 sd
							                01 = 40 tk 2 sd
									10 = 80 tk 1 sd
									11 = 80 tk 2 sd	*/
    label->sides = (label->flags & 0x01)?2:1;
    label->tracks = (label->flags & 0x02)?80:40;
    strncpy(label->label,&buffer[17],LABELSIZE);	/* textual disk label		*/
    label->res = hdos_int_unpack(&buffer[77]);		/* reserved			*/
    label->secs = buffer[79];				/* sectors per track (fixed 10)	*/

    return(TRUE);
}

/************************************************************************
 * NAME:	hdos_label_pack()
 *
 * DESCR:	Given the label structure, re-pack it into the given
 *		buffer.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
static int
hdos_label_pack(struct hdos_label *label, unsigned char *buffer)
{

    buffer[0] = label->serial & 0x00ff;			/* disk serial number		*/

    hdos_int_pack(					/* date disk initialized	*/
	hdos_date_pack(&label->initdate),&buffer[1]);

    hdos_int_pack(label->dirsec,&buffer[3]);		/* logical sec where dir starts	*/
    hdos_int_pack(label->grtsec,&buffer[5]);		/* logical sec where GRT starts	*/
    buffer[7] = label->grpsize & 0x00ff;		/* group size (2,4, or 8 secs)	*/
    buffer[8] = label->voltype & 0x00ff;		/* type of vol (0 = data,
							                1 = bootable,
									2 = no dir)	*/
    buffer[9] = label->version & 0x00ff;		/* version of init.abs for init	*/
    hdos_int_pack(label->rgtsec,&buffer[10]);		/* logical sec where RGT starts	*/
    hdos_int_pack(label->size,&buffer[12]);		/* number of sectors on disk	*/
    hdos_int_pack(label->physize,&buffer[14]);		/* phys sector size (fixed 256)	*/
    label->flags = (label->sides == 1)?0x00:0x01;
    label->flags |= (label->tracks == 40)?0x00:0x02;
    buffer[16] = label->flags & 0x00ff;			/* volume flags 00 = 40 tk 1 sd
							                01 = 40 tk 2 sd
									10 = 80 tk 1 sd
									11 = 80 tk 2 sd	*/
    strncpy(&buffer[17],label->label,LABELSIZE);	/* textual disk label		*/
    hdos_int_pack(label->res,&buffer[77]);		/* reserved			*/
    buffer[79] = label->secs & 0x00ff;			/* sectors per track (fixed 10)	*/

    return(TRUE);
}

/************************************************************************
 * NAME:	hdos_label_read()
 *
 * DESCR:	Reads the floppy label data from the file system.  Fills
 *		the label structure in the hdosfs structure.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
hdos_label_read(struct hdosfs *hdosfs)
{
    char	buffer[256];		/* one sector buffer		*/

    if (!hdos_getpsect(hdosfs,LABELTRACK,LABELSECTOR,buffer))
	return(FALSE);

    if (!hdos_label_unpack(&hdosfs->labelbuf,buffer)) 
	return(FALSE);

    /* some values need to be locked in because they appear to not be	*/
    /* consistently set correctly on some disks.			*/

    hdosfs->labelbuf.secs = 10;
    hdosfs->labelbuf.physize = 256;

    return(TRUE);
}

/************************************************************************
 * NAME:	hdos_label_write()
 *
 * DESCR:	Writes the label back out to the floppy.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
hdos_label_write(struct hdosfs *hdosfs)
{
    char	buffer[256];		/* one sector buffer		*/

    if (!hdos_label_pack(&hdosfs->labelbuf,buffer))
	return(FALSE);
    
    return(hdos_getpsect(hdosfs,LABELTRACK,LABELSECTOR,buffer));
}

/************************************************************************
 * NAME:	hdos_label_report()
 *
 * DESCR:	write a label report to the given buffer.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
char *
hdos_label_report(struct hdosfs *hdosfs, char *buffer, int verbosity)
{
    char 	*ptr = buffer;

    if (verbosity >= 2) {
	ptr += sprintf(ptr,"Init v%d, Serial %d, Label: %60s\n",
			  hdosfs->labelbuf.version,
			  hdosfs->labelbuf.serial,
			  hdosfs->labelbuf.label);
    }

    if (verbosity >= 3) {
	ptr += hdos_date_format(&hdosfs->labelbuf.initdate,ptr,0);
	ptr += sprintf(ptr,"\n");
	ptr += sprintf(ptr,"Disk with %d tracks and %d sectors (%d sides)\n",
			  HDOS_TRACKS(hdosfs),HDOS_SECTORS(hdosfs),HDOS_SIDES(hdosfs));
	ptr += sprintf(ptr,"dir start sect %d\n",HDOS_DIRLOC(hdosfs));
	ptr += sprintf(ptr,"GRT start sect %d\n",HDOS_GRTLOC(hdosfs));
	ptr += sprintf(ptr,"RGT start sect %d\n",HDOS_RGTLOC(hdosfs));
	ptr += sprintf(ptr,"group size %d\n",HDOS_GROUPSIZE(hdosfs));
	ptr += sprintf(ptr,"volume type %s\n",(hdosfs->labelbuf.voltype == 0)?"data":
			  (hdosfs->labelbuf.voltype == 1)?"bootable":"no dir");
    }

    return(buffer);
}


